<?php

namespace App\Http\Controllers;

use App\Models\ShortsVideo;
use App\Models\Category;
use App\Models\ShortsCategory;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Aws\S3\S3Client;
use Aws\Exception\AwsException;

class ShortsVideoController extends Controller
{
    private $r2Client;
    private $r2Enabled;
    
    public function __construct()
    {
        $this->initializeR2Client();
    }
    
    /**
     * Initialize Cloudflare R2 client
     */
    private function initializeR2Client()
    {
        try {
            $this->r2Enabled = Setting::getValue('r2_enabled') == '1';
            
            if ($this->r2Enabled) {
                $endpoint = Setting::getValue('r2_endpoint');
                $accessKey = Setting::getValue('r2_access_key');
                $secretKey = Setting::getValue('r2_secret_key');
                
                if (!empty($endpoint) && !empty($accessKey) && !empty($secretKey)) {
                    $this->r2Client = new S3Client([
                        'version' => 'latest',
                        'region' => 'auto',
                        'endpoint' => $endpoint,
                        'credentials' => [
                            'key' => $accessKey,
                            'secret' => $secretKey,
                        ],
                        'use_path_style_endpoint' => false,
                    ]);
                }
            }
        } catch (\Exception $e) {
            $this->r2Enabled = false;
        }
    }
    
    /**
     * Display a listing of the shorts videos
     */
    public function index(Request $request)
    {
        $query = ShortsVideo::query()->with('category');
        
        // Filter by category
        if ($request->has('category_id') && !empty($request->category_id)) {
            $query->where('category_id', $request->category_id);
        }
        
        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        // Filter by status
        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }
        
        $shorts = $query->orderBy('order')->orderBy('created_at', 'desc')->paginate(20);
        $categories = ShortsCategory::orderBy('orders', 'ASC')->get();
        
        return view('shorts.index', compact('shorts', 'categories'));
    }
    
    /**
     * Show the form for creating a new shorts video
     */
    public function create()
    {
        $categories = ShortsCategory::orderBy('orders', 'ASC')->get();
        return view('shorts.create', compact('categories'));
    }
    
    /**
     * Store a newly created shorts video
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'video' => 'required|file|mimes:mp4,mov,avi,mkv|max:204800', // 200MB max
            'thumbnail' => 'nullable|image|mimes:jpg,jpeg,png|max:5120', // 5MB max for thumbnail
            'category_id' => 'nullable|exists:shorts_category,id',
            'status' => 'nullable|in:active,inactive',
            'is_featured' => 'nullable|boolean',
            'order' => 'nullable|integer',
        ]);
        
        try {
            $videoUrl = '';
            $thumbnailUrl = '';
            $r2Key = '';
            
            // Upload video
            if ($request->hasFile('video') && $this->r2Enabled && $this->r2Client) {
                // Upload to Cloudflare R2
                $videoFile = $request->file('video');
                $r2Key = 'shorts/' . uniqid() . '_' . time() . '.' . $videoFile->getClientOriginalExtension();
                
                $result = $this->r2Client->putObject([
                    'Bucket' => Setting::getValue('r2_bucket'),
                    'Key' => $r2Key,
                    'Body' => fopen($videoFile->getRealPath(), 'r'),
                    'ContentType' => $videoFile->getMimeType(),
                    'CacheControl' => 'public, max-age=31536000',
                ]);
                
                $videoUrl = $this->getR2PublicUrl($r2Key);
            } else {
                // Fallback: Upload to local storage
                $videoFile = $request->file('video');
                $videoName = uniqid() . '_' . time() . '.' . $videoFile->getClientOriginalExtension();
                $videoFile->move(public_path('uploads/shorts'), $videoName);
                $videoUrl = url('uploads/shorts/' . $videoName);
            }
            
            // Handle thumbnail
            if ($request->hasFile('thumbnail')) {
                // User uploaded custom thumbnail - compress to 50% size
                $thumbnailUrl = $this->processAndUploadThumbnail($request->file('thumbnail'));
            } else {
                // Fallback: Use video URL as thumbnail (like existing video system)
                $thumbnailUrl = $videoUrl;
            }
            
            // Create shorts video record
            $short = ShortsVideo::create([
                'title' => $request->title,
                'description' => $request->description,
                'video_url' => $videoUrl,
                'thumbnail_url' => $thumbnailUrl,
                'r2_key' => $r2Key,
                'category_id' => $request->category_id,
                'status' => $request->status ?? 'active',
                'is_featured' => $request->is_featured ?? 0,
                'order' => $request->order ?? 0,
            ]);
            
            // Return JSON for AJAX requests
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Shorts video uploaded successfully!',
                    'data' => $short
                ], 200);
            }
            
            return redirect()->route('shorts.index')->with('success', 'Shorts video uploaded successfully!');
            
        } catch (\Exception $e) {
            // Return JSON error for AJAX requests
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 422);
            }
            
            return back()->with('error', 'Error uploading video: ' . $e->getMessage());
        }
    }
    
    /**
     * Show the form for editing the shorts video
     */
    public function edit($id)
    {
        $short = ShortsVideo::findOrFail($id);
        $categories = ShortsCategory::orderBy('orders', 'ASC')->get();
        return view('shorts.edit', compact('short', 'categories'));
    }
    
    /**
     * Update the shorts video
     */
    public function update(Request $request, $id)
    {
        $short = ShortsVideo::findOrFail($id);
        
        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'video' => 'nullable|file|mimes:mp4,mov,avi,mkv|max:204800',
            'thumbnail' => 'nullable|image|mimes:jpg,jpeg,png|max:5120', // 5MB max
            'category_id' => 'nullable|exists:shorts_category,id',
            'status' => 'nullable|in:active,inactive',
            'is_featured' => 'nullable|boolean',
            'order' => 'nullable|integer',
        ]);
        
        try {
            // Update video if new one is uploaded
            if ($request->hasFile('video')) {
                // Delete old video files (both R2 and local)
                $this->deleteVideoFiles($short);
                
                // Upload new video
                if ($this->r2Enabled && $this->r2Client) {
                    $videoFile = $request->file('video');
                    $r2Key = 'shorts/' . uniqid() . '_' . time() . '.' . $videoFile->getClientOriginalExtension();
                    
                    $this->r2Client->putObject([
                        'Bucket' => Setting::getValue('r2_bucket'),
                        'Key' => $r2Key,
                        'Body' => fopen($videoFile->getRealPath(), 'r'),
                        'ContentType' => $videoFile->getMimeType(),
                        'CacheControl' => 'public, max-age=31536000',
                    ]);
                    
                    $short->video_url = $this->getR2PublicUrl($r2Key);
                    $short->r2_key = $r2Key;
                } else {
                    $videoFile = $request->file('video');
                    $videoName = uniqid() . '_' . time() . '.' . $videoFile->getClientOriginalExtension();
                    $videoFile->move(public_path('uploads/shorts'), $videoName);
                    $short->video_url = url('uploads/shorts/' . $videoName);
                }
            }
            
            // Update thumbnail if new one is uploaded
            if ($request->hasFile('thumbnail')) {
                // Delete old thumbnail from R2 if exists
                $publicDomain = Setting::getValue('r2_public_domain');
                if ($short->thumbnail_url && $publicDomain && strpos($short->thumbnail_url, $publicDomain) !== false) {
                    try {
                        if ($this->r2Enabled && $this->r2Client) {
                            $urlParts = parse_url($short->thumbnail_url);
                            $thumbnailKey = ltrim($urlParts['path'], '/');
                            
                            $this->r2Client->deleteObject([
                                'Bucket' => Setting::getValue('r2_bucket'),
                                'Key' => $thumbnailKey,
                            ]);
                        }
                    } catch (\Exception $e) {
                        \Log::warning("Failed to delete old R2 thumbnail: " . $e->getMessage());
                    }
                }
                
                // Delete old thumbnail from local storage if exists
                if ($short->thumbnail_url && strpos($short->thumbnail_url, url('uploads/shorts/thumbnails')) === 0) {
                    $oldThumbnailPath = str_replace(url('uploads/shorts/thumbnails/'), '', $short->thumbnail_url);
                    $fullOldThumbnailPath = public_path('uploads/shorts/thumbnails/' . $oldThumbnailPath);
                    if (file_exists($fullOldThumbnailPath)) {
                        @unlink($fullOldThumbnailPath);
                    }
                }
                
                // Process and compress thumbnail to 50% size
                $short->thumbnail_url = $this->processAndUploadThumbnail($request->file('thumbnail'));
            } elseif ($request->hasFile('video')) {
                // If video changed but no thumbnail, use video URL as fallback
                $short->thumbnail_url = $short->video_url;
            }
            
            // Update other fields
            $short->title = $request->title;
            $short->description = $request->description;
            $short->category_id = $request->category_id;
            if ($request->has('status')) {
                $short->status = $request->status;
            }
            $short->is_featured = $request->is_featured ?? 0;
            $short->order = $request->order ?? $short->order;
            $short->save();
            
            // Return JSON for AJAX requests
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Shorts video updated successfully!',
                    'data' => $short
                ], 200);
            }
            
            return redirect()->route('shorts.index')->with('success', 'Shorts video updated successfully!');
            
        } catch (\Exception $e) {
            // Return JSON error for AJAX requests
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 422);
            }
            
            return back()->with('error', 'Error updating video: ' . $e->getMessage());
        }
    }
    
    /**
     * Remove the shorts video
     */
    public function destroy($id)
    {
        try {
            $short = ShortsVideo::findOrFail($id);
            
            // Try to delete video and thumbnail files from storage (but don't stop if it fails)
            try {
                $this->deleteVideoFiles($short);
            } catch (\Exception $e) {
                // Continue to database deletion even if file deletion fails
                \Log::warning("Failed to delete files for shorts video ID {$id}: " . $e->getMessage());
            }
            
            // Permanently delete database record (forceDelete removes it completely, not soft delete)
            $deleted = $short->forceDelete();
            
            if ($deleted) {
                return redirect()->route('shorts.index')->with('success', 'Shorts video deleted successfully!');
            } else {
                return back()->with('error', 'Failed to delete database record');
            }
            
        } catch (\Exception $e) {
            return back()->with('error', 'Error deleting video: ' . $e->getMessage());
        }
    }
    
    /**
     * Get R2 public URL
     */
    private function getR2PublicUrl($key)
    {
        $domain = Setting::getValue('r2_public_domain');
        return "https://{$domain}/{$key}";
    }
    
    /**
     * Delete video and thumbnail files from storage
     */
    private function deleteVideoFiles($short)
    {
        $bucket = Setting::getValue('r2_bucket');
        $publicDomain = Setting::getValue('r2_public_domain');
        
        // Delete video from R2
        if ($short->r2_key && $this->r2Enabled && $this->r2Client) {
            try {
                $this->r2Client->deleteObject([
                    'Bucket' => $bucket,
                    'Key' => $short->r2_key,
                ]);
            } catch (\Exception $e) {
                \Log::warning("Failed to delete video from R2: " . $e->getMessage());
            }
        }
        
        // Delete thumbnail from R2
        if ($short->thumbnail_url && $this->r2Enabled && $this->r2Client) {
            try {
                // Check if thumbnail is on R2 (contains public domain)
                if ($publicDomain && strpos($short->thumbnail_url, $publicDomain) !== false) {
                    // Extract R2 key from thumbnail URL
                    // Format: https://pub-abc.r2.dev/shorts/thumbnails/xxxxx_thumb.jpg
                    $urlParts = parse_url($short->thumbnail_url);
                    $thumbnailKey = ltrim($urlParts['path'], '/');
                    
                    $this->r2Client->deleteObject([
                        'Bucket' => $bucket,
                        'Key' => $thumbnailKey,
                    ]);
                }
            } catch (\Exception $e) {
                \Log::warning("Failed to delete thumbnail from R2: " . $e->getMessage());
            }
        }
        
        // Delete video from local storage
        try {
            if ($short->video_url && strpos($short->video_url, url('uploads/shorts')) === 0) {
                $videoPath = str_replace(url('uploads/shorts/'), '', $short->video_url);
                $fullVideoPath = public_path('uploads/shorts/' . $videoPath);
                if (file_exists($fullVideoPath)) {
                    @unlink($fullVideoPath);
                }
            }
        } catch (\Exception $e) {
            \Log::warning("Failed to delete local video: " . $e->getMessage());
        }
        
        // Delete thumbnail from local storage
        try {
            if ($short->thumbnail_url && strpos($short->thumbnail_url, url('uploads/shorts/thumbnails')) === 0) {
                $thumbnailPath = str_replace(url('uploads/shorts/thumbnails/'), '', $short->thumbnail_url);
                $fullThumbnailPath = public_path('uploads/shorts/thumbnails/' . $thumbnailPath);
                if (file_exists($fullThumbnailPath)) {
                    @unlink($fullThumbnailPath);
                }
            }
        } catch (\Exception $e) {
            \Log::warning("Failed to delete local thumbnail: " . $e->getMessage());
        }
    }
    
    /**
     * Test R2 connection
     */
    public function testR2Connection(Request $request)
    {
        try {
            // Use POST parameters if provided, otherwise use saved settings
            $endpoint = $request->r2_endpoint ?? Setting::getValue('r2_endpoint');
            $accessKey = $request->r2_access_key ?? Setting::getValue('r2_access_key');
            $secretKey = $request->r2_secret_key ?? Setting::getValue('r2_secret_key');
            $bucket = $request->r2_bucket ?? Setting::getValue('r2_bucket');
            
            if (empty($endpoint) || empty($accessKey) || empty($secretKey) || empty($bucket)) {
                return response()->json(['success' => false, 'message' => 'R2 credentials not provided']);
            }
            
            // Create temporary R2 client with provided credentials
            $testClient = new S3Client([
                'version' => 'latest',
                'region' => 'auto',
                'endpoint' => $endpoint,
                'credentials' => [
                    'key' => $accessKey,
                    'secret' => $secretKey,
                ],
                'use_path_style_endpoint' => false,
            ]);
            
            // Test connection by listing buckets
            $testClient->listBuckets();
            
            // Test if specific bucket exists
            $testClient->headBucket(['Bucket' => $bucket]);
            
            return response()->json(['success' => true, 'message' => 'R2 connection successful! Bucket verified.']);
            
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Connection failed: ' . $e->getMessage()]);
        }
    }
    
    /**
     * Process and upload thumbnail with 50% size reduction
     */
    private function processAndUploadThumbnail($thumbnailFile)
    {
        try {
            // Create temp file for compressed image
            $tempPath = sys_get_temp_dir() . '/' . uniqid() . '.jpg';
            
            // Get image info
            $imageInfo = getimagesize($thumbnailFile->getRealPath());
            $mimeType = $imageInfo['mime'];
            
            // Load image based on type
            switch ($mimeType) {
                case 'image/jpeg':
                    $sourceImage = imagecreatefromjpeg($thumbnailFile->getRealPath());
                    break;
                case 'image/png':
                    $sourceImage = imagecreatefrompng($thumbnailFile->getRealPath());
                    break;
                case 'image/gif':
                    $sourceImage = imagecreatefromgif($thumbnailFile->getRealPath());
                    break;
                default:
                    $sourceImage = imagecreatefromjpeg($thumbnailFile->getRealPath());
            }
            
            if (!$sourceImage) {
                // If image processing fails, upload original
                return $this->uploadThumbnailFile($thumbnailFile);
            }
            
            // Get original dimensions
            $originalWidth = imagesx($sourceImage);
            $originalHeight = imagesy($sourceImage);
            
            // Calculate 50% dimensions
            $newWidth = intval($originalWidth * 0.5);
            $newHeight = intval($originalHeight * 0.5);
            
            // Create new image with 50% size
            $resizedImage = imagecreatetruecolor($newWidth, $newHeight);
            
            // Preserve transparency for PNG
            if ($mimeType === 'image/png') {
                imagealphablending($resizedImage, false);
                imagesavealpha($resizedImage, true);
            }
            
            // Resize image
            imagecopyresampled(
                $resizedImage, $sourceImage,
                0, 0, 0, 0,
                $newWidth, $newHeight,
                $originalWidth, $originalHeight
            );
            
            // Save compressed image as JPEG (quality 85)
            imagejpeg($resizedImage, $tempPath, 85);
            
            // Free memory
            imagedestroy($sourceImage);
            imagedestroy($resizedImage);
            
            // Upload the compressed image
            $thumbnailName = uniqid() . '_' . time() . '_thumb.jpg';
            
            if ($this->r2Enabled && $this->r2Client) {
                // Upload to R2
                $thumbnailKey = 'shorts/thumbnails/' . $thumbnailName;
                
                $this->r2Client->putObject([
                    'Bucket' => Setting::getValue('r2_bucket'),
                    'Key' => $thumbnailKey,
                    'Body' => fopen($tempPath, 'r'),
                    'ContentType' => 'image/jpeg',
                    'CacheControl' => 'public, max-age=31536000',
                ]);
                
                // Delete temp file
                @unlink($tempPath);
                
                return $this->getR2PublicUrl($thumbnailKey);
            } else {
                // Upload to local storage
                $uploadPath = public_path('uploads/shorts/thumbnails');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0777, true);
                }
                
                copy($tempPath, $uploadPath . '/' . $thumbnailName);
                
                // Delete temp file
                @unlink($tempPath);
                
                return url('uploads/shorts/thumbnails/' . $thumbnailName);
            }
            
        } catch (\Exception $e) {
            // If compression fails, upload original
            return $this->uploadThumbnailFile($thumbnailFile);
        }
    }
    
    /**
     * Upload thumbnail file without processing
     */
    private function uploadThumbnailFile($thumbnailFile)
    {
        $thumbnailName = uniqid() . '_' . time() . '.jpg';
        
        if ($this->r2Enabled && $this->r2Client) {
            $thumbnailKey = 'shorts/thumbnails/' . $thumbnailName;
            
            $this->r2Client->putObject([
                'Bucket' => Setting::getValue('r2_bucket'),
                'Key' => $thumbnailKey,
                'Body' => fopen($thumbnailFile->getRealPath(), 'r'),
                'ContentType' => $thumbnailFile->getMimeType(),
                'CacheControl' => 'public, max-age=31536000',
            ]);
            
            return $this->getR2PublicUrl($thumbnailKey);
        } else {
            $uploadPath = public_path('uploads/shorts/thumbnails');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }
            
            $thumbnailFile->move($uploadPath, $thumbnailName);
            return url('uploads/shorts/thumbnails/' . $thumbnailName);
        }
    }
    
    /**
     * Toggle shorts status
     */
    public function toggleStatus(Request $request)
    {
        $short = ShortsVideo::findOrFail($request->id);
        $short->status = $request->checked ? 'active' : 'inactive';
        $short->save();
        
        return response()->json(['success' => true]);
    }
    
    /**
     * Toggle featured status
     */
    public function toggleFeatured(Request $request)
    {
        $short = ShortsVideo::findOrFail($request->id);
        $short->is_featured = $request->featured;
        $short->save();
        
        return response()->json(['success' => true]);
    }
    
    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $ids = explode(',', $request->shorts_ids);
        $action = $request->action_type;
        
        if (empty($ids)) {
            return back()->with('error', 'No shorts selected');
        }
        
        switch ($action) {
            case 'enable':
                ShortsVideo::whereIn('id', $ids)->update(['status' => 'active']);
                return back()->with('success', 'Shorts activated successfully');
                
            case 'disable':
                ShortsVideo::whereIn('id', $ids)->update(['status' => 'inactive']);
                return back()->with('success', 'Shorts deactivated successfully');
                
            case 'delete':
                $deletedCount = 0;
                foreach ($ids as $id) {
                    $short = ShortsVideo::find($id);
                    if ($short) {
                        // Try to delete video and thumbnail files from storage
                        try {
                            $this->deleteVideoFiles($short);
                        } catch (\Exception $e) {
                            // Continue to database deletion even if file deletion fails
                            \Log::warning("Failed to delete files for shorts video ID {$id}: " . $e->getMessage());
                        }
                        
                        // Permanently delete database record (forceDelete removes it completely)
                        if ($short->forceDelete()) {
                            $deletedCount++;
                        }
                    }
                }
                return back()->with('success', "{$deletedCount} shorts deleted successfully");
                
            default:
                return back()->with('error', 'Invalid action');
        }
    }
}
