<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ShortsVideo extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'shorts_videos';

    protected $fillable = [
        'title',
        'description',
        'video_url',
        'thumbnail_url',
        'r2_key',
        'duration',
        'views',
        'likes',
        'shares',
        'category_id',
        'status',
        'is_featured',
        'order',
        'metadata'
    ];

    protected $casts = [
        'metadata' => 'array',
        'is_featured' => 'boolean',
        'views' => 'integer',
        'likes' => 'integer',
        'shares' => 'integer',
        'duration' => 'integer',
        'order' => 'integer',
    ];

    protected $dates = ['deleted_at'];

    /**
     * Get the category that owns the shorts video
     */
    public function category()
    {
        return $this->belongsTo(ShortsCategory::class, 'category_id');
    }

    /**
     * Increment view count
     */
    public function incrementViews()
    {
        $this->increment('views');
        return $this;
    }

    /**
     * Increment like count
     */
    public function incrementLikes()
    {
        $this->increment('likes');
        return $this;
    }

    /**
     * Decrement like count
     */
    public function decrementLikes()
    {
        $this->decrement('likes');
        return $this;
    }

    /**
     * Increment share count
     */
    public function incrementShares()
    {
        $this->increment('shares');
        return $this;
    }

    /**
     * Scope a query to only include active videos
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope a query to only include featured videos
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to order by custom order and creation date
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('order', 'asc')->orderBy('created_at', 'desc');
    }
}
